{ ********************************************************************************** }
{ ****** This unit overrides the graphs unit to draw a line graph of the data ****** }
{ ********************************************************************************** }
{ **** This component was written and created by Keith Moore,                   **** }
{ ****                                           Process Automation Dept.       **** }
{ ****                                           UCB Films plc.                 **** }
{ ****                                           Wigton                         **** }
{ ********************************************************************************** }

unit Lgraphs;

interface

uses
  SysUtils, WinTypes, WinProcs, Classes, Graphics, Controls,
  Forms, Dialogs, Graphs, Printers, Menus;

type
  TSelectMode = (smManual, smAuto);
  TCustomLineGraph = Class( TCustomGraph )
  private
    { **** Auto Select Variables **** }
    MouseSelRect : TRect;                                             { Selected region in pixels }
    MouseSelect  : Boolean;
    MouseOrigin  : TPoint;                       { Point where the mouse button was first pressed }

    { **** Zoom Variables **** }
    Zoomed     : Byte;                                                      { Is the graph zoomed }
    GXValues,                                                             { Original graph values }
    GYValues   : Byte;
    GXMaxValue,
    GXMinValue,
    GYMaxValue,
    GYMinValue : Double;

    { **** Selction Variables **** }
    Selected : Boolean;                          { Is there a region selected True = Y, False = N }
    SelXMax,
    SelXMin,
    SelYMax,
    SelYMin  : Double;

    { **** Property variables **** }
    FYData, FXData : TDoubleStream;
    FLineColor     : TColor;
    FMarkers       : Boolean;                                  { Place markers at points on graph }
    FSelectMode    : TSelectMode;

    { **** Property procedures **** }
    procedure SetXData(Value : TDoubleStream);
    procedure SetYData(Value : TDoubleStream);
    procedure SetLineColor(Value : TColor);
    procedure SetMarkers(Value : Boolean);
    procedure SetSelectMode(Value : TSelectMode);
  protected
    { **** Zooming and Selection routines **** }
    procedure ZoomIn; virtual;
    procedure ZoomOut; virtual;
    procedure DrawSelection; virtual;                        { Draws the selected area (inverted) }
    procedure SelectPixels(SelRect : TRect); virtual;              { Selects an area using pixels }
    procedure SelectValues(XMin, XMax, YMin, YMax : Double); virtual;    { Selects an areas values}
    procedure ClearSelection; virtual;                             { Clears the current selection }
    function ValuesToRect(XMin, XMax, YMin, YMax : Double) : TRect; virtual;
    procedure RectToValues(TR : TRect; var XMin, XMax, YMin, YMax : Double); virtual;
    procedure ScaleValues; virtual;                            { Scales values to a sensible d.p. }
    function LengthOfFraction(Value : Double) : Byte;

    { **** Auto Selection routines **** }
    procedure MouseDown(Button : TMouseButton; Shift : TShiftState; X, Y : Integer); override;
    procedure MouseMove(Shift : TShiftState; X, Y : Integer); override;
    procedure MouseUp(Button : TMouseButton; Shift : TShiftState; X, Y : Integer); override;

    { **** Printing Procedures **** }
    procedure PrintDataLines(VP, HP : Single); override;

    { **** Painting Procedures **** }
    procedure Paint; override;
    procedure GetCalibration(var VP, HP, SX, EX, Step : Single); override;
    procedure PaintDataLines(var AnImage : TBitmap; VP, HP : Single); override;

    { ******** Line Graph properties ********* }
    property XData : TDoubleStream read FXData write SetXData;
    property YData : TDoubleStream read FYData write SetYData;
    property LineColor : TColor read FLineColor write SetLineColor;
    property Markers : Boolean read FMarkers write SetMarkers;
    property SelectMode : TSelectMode read FSelectMode write SetSelectMode;

    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  end;

  TLineGraph = Class( TCustomLineGraph )
  public
    property XData;                                     { Allow user access to co-ordinate arrays }
    property YData;

    constructor Create(AOwner : TComponent); override;
    { **** Printing Routines **** }
    procedure PrintGraph; override;
    { **** Updating Graph **** }
    procedure DisableGraph;                       { Procedure that stop the graph from repainting }
    procedure EnableGraph;                                                   { Enables repainting }
    { **** Selection Routines **** }
    procedure ZoomIn; override;                                      { Zoom in on selected region }
    procedure ZoomOut; override;                                          { Zoom out to full size }
    procedure SelectPixels(SelRect : TRect); override;
    procedure SelectValues(XMin, XMax, YMin, YMax : Double); override;
    procedure ClearSelection; override;
  published
    property Align default alNone;                                       { Publish the properties }
    property AxisColor default clBlack;
    property BorderStyle default bsSingle;
    property Color default clWhite;
    property Cursor default crCross;
    property DragCursor;
    property DragMode;
    property Font;
    property LineColor default clBlue;
    property Markers default false;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PlotAreaBorderStyle default bsNone;
    property PopupMenu;
    property PrintLeft default 0;
    property PrintTop default 0;
    property PrintHeight default 1000;
    property PrintWidth default 1000;
    property SelectMode default smManual;
    property ShowHint;
    property X_Axis;
    property Y_Axis;

    property OnClick;                                                        { Publish the events }
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

function Max(Value1, Value2 : Extended) : Extended;
function Min(Value1, Value2 : Extended) : Extended;
function Exp10(Value : Double; Multiplier : Integer) : Double;
procedure Register;                                          { Register the component with delphi }

implementation

{ ******************************************************* }
{ ************* TCustomLineGraph procedures ************* }
{ ******************************************************* }

constructor TCustomLineGraph.Create(AOwner : TComponent);
begin
  Inherited Create( AOwner );                                      { Initialize control variables }
  FXData := TDoubleStream.Create;
  FYData := TDoubleStream.Create;
  FXData.Clear;
  FYData.Clear;
  Zoomed := 0;
  Selected := False;
  MouseSelect := False;
end;

destructor TCustomLineGraph.Destroy;
begin
  FXData.Free;                                                              { Destroy Data arrays }
  FYData.Free;
  Inherited Destroy;
end;

procedure TCustomLineGraph.SetXData(Value : TDoubleStream);
begin
  If FXData <> Value Then
  begin
    FXData := Value;                                            { Whenever any data is changed... }
    RePaint;                                                               { ...Repaint the graph }
  end;
end;

procedure TCustomLineGraph.SetYData(Value : TDoubleStream);
begin
  If FYData <> Value Then
  begin
    FYData := Value;                                            { Whenever any data is changed... }
    RePaint;                                                               { ...Repaint the graph }
  end;
end;

procedure TCustomLineGraph.SetLineColor(Value : TColor);
begin
  If FLineColor <> Value Then
  begin
    FLineColor := Value;                                        { Whenever any data is changed... }
    RePaint;                                                               { ...Repaint the graph }
  end;
end;

procedure TCustomLineGraph.SetMarkers(Value : Boolean);
begin
  If FMarkers <> Value then
  begin
    FMarkers := Value;                                          { Whenever any data is changed... }
    RePaint;                                                               { ...Repaint the graph }
  end;
end;

procedure TCustomLineGraph.SetSelectMode(Value : TSelectMode);
begin
  If FSelectMode <> Value then
    FSelectMode := Value;
end;

{ **** Gets Pixels per value **** }
procedure TCustomLineGraph.GetCalibration(var VP, HP, SX, EX, Step : Single);
var
  GraphHeight, GraphWidth : Integer;

begin
  GraphWidth := GraphRect.Right - GraphRect.Left;                        { Width of plotting area }
  GraphHeight := GraphRect.Bottom - GraphRect.Top;                      { Height of plotting area }
  With Y_Axis Do
    VP := GraphHeight / (RealMaxValue - MinValue);                   { Calculate pixels per value }
  With X_Axis Do
    HP := GraphWidth / (RealMaxValue - MinValue);
  If FYData.Size > 1 then
    Step := (EX - SX) / (FYData.Size - 1)                   { Calculate the step between readings }
  Else
    Step := -1;
end;

{ **** Plots the data lines on the graph **** }
procedure TCustomLineGraph.PaintDataLines(var AnImage : TBitmap; VP, HP : Single);
var
  Loop                    : LongInt;
  ActualX, ActualY        : Single;                               { Actual postion of co-ordinate }
  CurrentX, CurrentY,                                            { Rounded postion of co-ordinate }
  GraphHeight, GraphWidth : Integer;

begin
  If (FYData.Size > 1) And (FXData.Size > 1) then               { If there is any data to plot... }
  begin                                                                  { ...draw the data lines }
    GraphHeight := GraphRect.Bottom - GraphRect.Top;
    With AnImage.Canvas.Pen Do                                                    { Setup the pen }
    begin
      Color := FLineColor;
      Mode := pmCopy;
      Style := psSolid;
      Width := 1;
    end;
    With AnImage.Canvas.Brush Do                                                { Setup the brush }
    begin
      Color := FLineColor;
      Style := bsSolid;
    end;
    ActualX := (FXData.Read( 1 ) - X_Axis.MinValue) * HP;           { Get first exact co-ordinate }
    ActualY := (FYData.Read( 1 ) - Y_Axis.MinValue) * VP;
    CurrentX := Round( ActualX );                                 { Get first rounded co-ordinate }
    CurrentY := Round(GraphHeight - ActualY);
    try
      With AnImage.Canvas Do
      begin
        If Markers = True then                                    { if markers are to be drawn... }
          FillRect( Rect(CurrentX - 2, CurrentY - 3, CurrentX + 2, CurrentY + 1) );  { ...draw it }
        MoveTo(CurrentX, CurrentY - 1);                        { Move cursor to first co-ordinate }
        For Loop := 2 To Round( Min(FXData.Size, FYData.Size) ) Do            { Plot all the data }
        begin
          ActualX := (FXData.Read( Loop ) - X_Axis.MinValue) * HP;       { Get next actual co-ord }
          ActualY := (FYData.Read( Loop ) - Y_Axis.MinValue) * VP;
          CurrentX := Round( ActualX );                                 { Get next rounded co-ord }
          CurrentY := Round(GraphHeight - ActualY);
          If Markers = True then                                          { Draw marker if needed }
            FillRect( Rect(CurrentX - 2, CurrentY - 3, CurrentX + 2, CurrentY + 1) );
          LineTo(CurrentX, CurrentY - 1);                         { Draw line to next co-ordinate }
        end;
      end;
    except
      On Exception Do                                                         { Report any errors }
        MessageDlg('An Error occured while trying to draw the graph', mtInformation, [mbOK], 0);
    end;
  end;
end;

{ **** Called when ever the graph needs repainted **** }
procedure TCustomLineGraph.Paint;
begin
  If RePaintGraph = True then                                { If the graph is in repaint mode... }
  begin
    try
      Inherited Paint;                                                            { ...repaint it }
      If Selected = True then                                  { If there is a selected region... }
        DrawSelection;                                              { ...draw the selected region }
    except
      On Exception Do                                    { If an error occured when zooming in... }
        ZoomOut;                                                                    { ...zoom out }
    end;
  end;
end;

{ **** Printing Routines **** }

procedure TCustomLineGraph.PrintDataLines(VP, HP : Single);
var
  Loop                            : LongInt;
  ActualX, ActualY                : Single;
  CurrentX, CurrentY, GraphHeight : Integer;
  OldClipRect                     : TRect;
  NewRegion                       : HRgn;

begin
  OldClipRect := Printer.Canvas.ClipRect;      { Save the original clipping region of the printer }
  ExcludeClipRect(Printer.Handle, 0, 0,             { Exclude any region apart from the plot area }
                  Printer.Canvas.ClipRect.Right, GraphRect.Top - 1);
  ExcludeClipRect(Printer.Handle, 0, GraphRect.Top, GraphRect.Left - 1, GraphRect.Bottom);
  ExcludeClipRect(Printer.Handle, GraphRect.Right + 1, GraphRect.Top,
                  Printer.Canvas.ClipRect.Right, GraphRect.Bottom);
  ExcludeClipRect(Printer.Handle, 0, GraphRect.Bottom + 1,
                  Printer.Canvas.ClipRect.Right, Printer.Canvas.ClipRect.Bottom);
  If (FYData.Size > 1) And (FXData.Size > 1) then               { If there is any data to plot... }
  begin                                                                  { ...draw the data lines }
    GraphHeight := GraphRect.Bottom - GraphRect.Top;
    with Printer.Canvas.Pen Do                                                    { setup the pen }
    begin
      Color := FLineColor;
      Width := 7;
    end;
    with Printer.Canvas.Brush Do                                                { setup the brush }
    begin
      Color := FLineColor;
      Style := bsSolid;
    end;
    ActualX := (FXData.Read( 1 ) - X_Axis.MinValue) * HP;               { Get first actual co-ord }
    ActualY := (FYData.Read( 1 ) - Y_Axis.MinValue) * VP;
    CurrentX := GraphRect.Left + Round( ActualX );                     { Get first rounded co-ord }
    CurrentY := GraphRect.Top + Round(GraphHeight - ActualY);
    try
      with Printer.Canvas do
      begin
        MoveTo(CurrentX, CurrentY - 1);                             { Move cursor to first co-ord }
        If Markers = True then                                            { Draw marker if needed }
          FillRect( Rect(CurrentX - 16, CurrentY - 15, CurrentX + 16, CurrentY + 17) );
        For Loop := 2 To Round( Min(FXData.Size, FYData.Size) ) Do             { Plot all co-ords }
        begin
          ActualX := (FXData.Read( Loop ) - X_Axis.MinValue) * HP;       { Get next actual co-ord }
          ActualY := (FYData.Read( Loop ) - Y_Axis.MinValue) * VP;
          CurrentX := GraphRect.Left + Round( ActualX );                { Get next rounded co-ord }
          CurrentY := GraphRect.Top + Round(GraphHeight - ActualY);
          LineTo(CurrentX, CurrentY - 1);                              { Draw line to next co-ord }
          If Markers = True then                                          { Draw marker if needed }
            FillRect( Rect(CurrentX - 16, CurrentY - 15, CurrentX + 16, CurrentY + 17) );
        end;
      end;
      NewRegion := CreateRectRgnIndirect( OldClipRect );       { Restore original clipping region }
      SelectClipRgn(Printer.Handle, NewRegion);
      DeleteObject( NewRegion );
    except
      On Exception Do                                                         { Report any errors }
      begin
        MessageDlg('An Error occured while trying to print the graph', mtInformation,
                   [mbOK], 0);
        abort;
      end;
    end;
  end;
end;

{ **** Zooming and Selection routines **** }
procedure TCustomLineGraph.ZoomIn;
var
  Temp : TNotifyEvent;

begin
  If Zoomed = 0 then                                      { If the graph is not already zoomed... }
  begin
    GXValues := X_Axis.Values;                                         { ...save the graph values }
    GXMaxValue := X_Axis.MaxValue;
    GXMinValue := X_Axis.MinValue;
    GYValues := Y_Axis.Values;
    GYMaxValue := Y_Axis.MaxValue;
    GYMinValue := Y_Axis.MinValue;
  end;
  If (Selected = True) And (Zoomed < 5) then                   { If there is a selected region... }
  begin
    with X_Axis Do
    begin
      Temp := OnValueChange;                                        { ...Stop the graph redrawing }
      OnValueChange := nil;
      Values := 2;                                                            { Setup axis values }
      MaxValue := SelXMax;
      MinValue := SelXMin;
      OnValueChange := Temp;                                            { Restore graph redrawing }
    end;
    with Y_Axis Do
    begin
      Temp := OnValueChange;                                        { ...Stop the graph redrawing }
      OnValueChange := nil;
      Values := 2;                                                            { Setup axis values }
      MaxValue := SelYMax;
      MinValue := SelYMin;
      OnValueChange := Temp;                                            { Restore graph redrawing }
    end;
    Selected := False;                                                   { Cancel selected region }
    Inc( Zoomed );
    RePaint;                                                                  { Re draw the graph }
  end;
end;

procedure TCustomLineGraph.ZoomOut;
var
  Temp : TNotifyEvent;

begin
  If Zoomed > 0 then                                         { If the graph has been zoomed in... }
  begin
    with X_Axis Do
    begin
      Temp := OnValueChange;                                   { ...restore original graph values }
      OnValueChange := nil;
      Values := GXValues;
      MaxValue := GXMaxValue;
      MinValue := GXMinValue;
      OnValueChange := Temp;
    end;
    with Y_Axis Do
    begin
      Temp := OnValueChange;
      OnValueChange := nil;
      Values := GYValues;
      MaxValue := GYMaxValue;
      MinValue := GYMinValue;
      OnValueChange := Temp;
    end;
    Selected := False;                                               { Cancel the selected region }
    Zoomed := 0;                                                     { Cancel the zoomed register }
    RePaint;                                                                  { Re draw the graph }
  end;
end;

{ **** Draws the currently selected region, inverted in color **** }
procedure TCustomLineGraph.DrawSelection;
var
  SelRect : TRect;

begin
  SelRect := ValuesToRect(SelXMin, SelXMax, SelYMin, SelYMax);    { Convert values to a rectangle }
  If (SelRect.Left <> SelRect.Right) And (SelRect.Top <> SelRect.Bottom) then
  begin
    Canvas.CopyMode := cmNotSrcCopy;                          { Copy mode is to invert the source }
    Canvas.CopyRect(SelRect, Canvas, SelRect);                          { Draw the inverted image }
  End;
end;

{ **** Select a region using pixels **** }
procedure TCustomLineGraph.SelectPixels(SelRect : TRect);
begin
  If Selected = True then                              { If there is a region already selected... }
    ClearSelection;                                               { ...Clear that selected region }
  If SelRect.Left < GraphRect.Left Then            { setup the values to give a correct rectangle }
    SelRect.Left := GraphRect.Left;
  If SelRect.Top < GraphRect.Top Then
    SelRect.Top := GraphRect.Top;
  If SelRect.Right > GraphRect.Right Then
    SelRect.Right := GraphRect.Right;
  If SelRect.Bottom > GraphRect.Bottom Then
    SelRect.Bottom := GraphRect.Bottom;
  RectToValues(SelRect, SelXMin, SelXMax, SelYMin, SelYMax);    { Convert the rectangle to values }
  ScaleValues;                                                  { Give the values a sensible d.p. }
  DrawSelection;                                                   { Draw the new selected region }
  Selected := True;
end;

{ **** Select an area with values from the graph **** }
procedure TCustomLineGraph.SelectValues(XMin, XMax, YMin, YMax : Double);
var
  SelRect : TRect;

begin
  If Selected = True then                                { If there is already a selected area... }
    ClearSelection;                                                          { ...clear that area }
  If XMin < X_Axis.MinValue then                          { Set up values to be correct rectangle }
    XMin := X_Axis.MinValue;
  If XMax > X_Axis.RealMaxValue then
    XMax := X_Axis.RealMaxValue;
  If YMin < Y_Axis.MinValue then
    YMin := Y_Axis.MinValue;
  If YMax > Y_Axis.RealMaxValue then
    YMax := Y_Axis.RealMaxValue;
  SelXMin := XMin;
  SelXMax := XMax;
  SelYMin := YMin;
  SelYMax := YMax;
  DrawSelection;                                                   { Draw the new selected region }
  Selected := True;
end;

{ **** Clear away any previously selected region **** }
procedure TCustomLineGraph.ClearSelection;
var
  SelRect : TRect;

begin
  If Selected = True then                                         { If there is a selected region }
  begin
    SelRect := ValuesToRect(SelXMin, SelXMax, SelYMin, SelYMax);                  { Get rectangle }
    Canvas.CopyMode := cmNotSrcCopy;                                 { Invert the inverted region }
    Canvas.CopyRect(SelRect, Canvas, SelRect);
    Selected := False;
  end;
end;

{ **** Convert the graph values into pixels on the graph **** }
function TCustomLineGraph.ValuesToRect(XMin, XMax, YMin, YMax : Double) : TRect;
var
  HPixelsPerValue, VPixelsPerValue,
  SelWidth, SelHeight, InsetX, InsetY : Double;
  GraphWidth, GraphHeight             : Integer;

begin
  GraphWidth := GraphRect.Right - GraphRect.Left;               { Get dimensions of plotting area }
  GraphHeight := GraphRect.Bottom - GraphRect.Top;
  HPixelsPerValue := GraphWidth / (X_Axis.RealMaxValue - X_Axis.MinValue); { Get pixels per value }
  VPixelsPerValue := GraphHeight / (Y_Axis.RealMaxValue - Y_Axis.MinValue);
  SelWidth := (XMax - XMin) * HPixelsPerValue;            { Calculate selected regions dimensions }
  SelHeight := (YMax - YMin) * VPixelsPerValue;
  InsetX := (XMin - X_Axis.MinValue) * HPixelsPerValue;       { Calculate selected regions offset }
  InsetY := (Y_Axis.RealMaxValue - YMax) * VPixelsPerValue;
  Result.Left := Round(InsetX + GraphRect.Left);                    { Calculate co-ords in pixels }
  Result.Top := Round(InsetY + GraphRect.Top);
  Result.Right := Round(Result.Left + SelWidth);
  Result.Bottom := Round(Result.Top + SelHeight);
end;

{ **** Convert graph pixel locations into graph values **** }
procedure TCustomLineGraph.RectToValues(TR : TRect; var XMin, XMax, YMin, YMax : Double);
var
  HPixelsPerValue, VPixelsPerValue,
  SelWidth, SelHeight, InsetX, InsetY : Double;
  GraphWidth, GraphHeight             : Integer;

begin
  GraphWidth := GraphRect.Right - GraphRect.Left;                      { Get plot area dimensions }
  GraphHeight := GraphRect.Bottom - GraphRect.Top;
  SelWidth := TR.Right - TR.Left;                                  { Get selected area dimensions }
  SelHeight := TR.Bottom - TR.Top;
  HPixelsPerValue := GraphWidth / (X_Axis.RealMaxValue - X_Axis.MinValue); { Get pixels per value }
  VPixelsPerValue := GraphHeight / (Y_Axis.RealMaxValue - Y_Axis.MinValue);
  InsetX := TR.Left - GraphRect.Left;                                 { Get selected area offsets }
  InsetY := TR.Top - GraphRect.Top;
  XMin := (InsetX / HPixelsPerValue) + X_Axis.MinValue;            { Calculate max and min values }
  XMax := (SelWidth / HPixelsPerValue) + XMin;
  YMax := Y_Axis.RealMaxValue - (InsetY / VPixelsPerValue);
  YMin := YMax - (SelHeight / VPixelsPerValue);
end;

{ **** When zooming in from a pixel selection, the decimal places of the values become  **** }
{ **** stupidly large so it is neccassary to only allow 1 extra decimal place with each **** }
{ **** zoom.  This procedure handles this process                                       **** }
procedure TCustomLineGraph.ScaleValues;
var
  TempStr                                : String;
  XDP, YDP                               : Byte;

begin
  XDP := LengthOfFraction( X_Axis.RealMaxValue );        { Calculate the number of decimal places }
  If LengthOfFraction( X_Axis.MinValue ) > XDP then
    XDP := LengthOfFraction( X_Axis.MinValue );                     { Get largest number of d.p's }
  YDP := LengthOfFraction( Y_Axis.RealMaxValue );        { Calculate the number of decimal places }
  If LengthOfFraction( Y_Axis.MinValue ) > YDP then
    YDP := LengthOfFraction( Y_Axis.MinValue );                     { Get largest number of d.p's }
  Inc( XDP );                                                      { Allow one more decimal place }
  Inc( YDP );
  SelXMax := Exp10(Round( Exp10(SelXMax, XDP) ), -XDP);{ Calculate the new, more sensible, values }
  SelXMin := Exp10(Round( Exp10(SelXMin, XDP) ), -XDP);
  SelYMax := Exp10(Round( Exp10(SelYMax, YDP) ), -YDP);
  SelYMin := Exp10(Round( Exp10(SelYMin, YDP) ), -YDP);
end;

{ **** Returns the number of decimal places within the given argument **** }
function TCustomLineGraph.LengthOfFraction(Value : Double) : Byte;
var
  TempStr : String;

begin
  TempStr := FloatToStr( Value );                                 { Convert the value to a string }
  If Pos('.', TempStr) = 0 then                                        { Locate the decimal point }
    Result := 0                            { If there is no decimal point then there are no d.p's }
  Else                                                                               { ...else... }
  begin
    Delete(TempStr, 1, Pos('.', TempStr));      { Delete numbers to the left of the decimal point }
    Result := Length( TempStr );                               { Get the number of decimal places }
  end;
end;

{ ************* Auto Selection Routines ************** }
procedure TCustomLineGraph.MouseDown(Button : TMouseButton; Shift : TShiftState; X, Y : Integer);
var
  TempPtr : TMouseEvent;

begin
  If Button = mbLeft then                                { If the left mouse button is pressed... }
    If FSelectMode = smAuto then                          { ...and auto select mode is enabled... }
    begin
      MouseSelect := True;                                           { ...Set the selected region }
      MouseOrigin := Point(X, Y);
    end;
  TempPtr := OnMouseDown;
  If Assigned( TempPtr ) then
    OnMouseDown(Self, Button, Shift, X, Y);
end;

procedure TCustomLineGraph.MouseMove(Shift : TShiftState; X, Y : Integer);
var
  TempPtr : TMouseMoveEvent;
  Temp    : Integer;

begin
  If (FSelectMode = smAuto) And (MouseSelect = True)  then     { If a selection hase beem made... }
  begin
    MouseSelRect.Left := Round( Min(X, MouseOrigin.X) );                  { Get the selected area }
    MouseSelRect.Right := Round( Max(X, MouseOrigin.X) );
    MouseSelRect.Top := Round( Min(Y, MouseOrigin.Y) );
    MouseSelRect.Bottom := Round( Max(Y, MouseOrigin.Y) );
    With MouseSelRect Do
    begin
      If (Left < GraphRect.Left) Or (Right > GraphRect.Right) Then   { mouse goes out of range... }
      begin
        Left := GraphRect.Left;                                            { Select an entire row }
        Right := GraphRect.Right;
      end;
      If (Top < GraphRect.Top) Or (Bottom > GraphRect.Bottom) Then
      begin
        Top := GraphRect.Top;                                           { Select an entire column }
        Bottom := GraphRect.Bottom;
      end;
    end;
    SelectPixels( MouseSelRect );                                      { Draw the selected region }
  end;
  TempPtr := OnMouseMove;
  If Assigned( TempPtr ) then
    OnMouseMove(Self, Shift, X, Y);
end;

procedure TCustomLineGraph.MouseUp(Button : TMouseButton; Shift : TShiftState; X, Y : Integer);
var
  TempPtr : TMouseEvent;

begin
  If (FSelectMode = smAuto) And (MouseSelect = True) then
  begin
    MouseSelRect.Left := Round( Min(X, MouseOrigin.X) );                  { Get the selected area }
    MouseSelRect.Right := Round( Max(X, MouseOrigin.X) );
    MouseSelRect.Top := Round( Min(Y, MouseOrigin.Y) );
    MouseSelRect.Bottom := Round( Max(Y, MouseOrigin.Y) );
    With MouseSelRect Do
    begin
      If (Left < GraphRect.Left) Or (Right > GraphRect.Right) Then   { mouse goes out of range... }
      begin
        Left := GraphRect.Left;                                            { Select an entire row }
        Right := GraphRect.Right;
      end;
      If (Top < GraphRect.Top) Or (Bottom > GraphRect.Bottom) Then
      begin
        Top := GraphRect.Top;                                           { Select an entire column }
        Bottom := GraphRect.Bottom;
      end;
    End;
    MouseSelect := False;
    SelectPixels( MouseSelRect );                                      { Draw the selected region }
  end;
  TempPtr := OnMouseUp;
  If Assigned( TempPtr ) then
    OnMouseUp(Self, Button, Shift, X, Y);
end;

{ ************************************************* }
{ ************* TLineGraph procedures ************* }
{ ************************************************* }

constructor TLineGraph.Create(AOwner : TComponent);
begin
  Inherited Create( AOwner );
  AxisColor := clBlack;                                                  { Set the default values }
  Color := clWhite;
  Cursor := crCross;
  BorderStyle := bsSingle;
  Markers := False;
  PlotAreaBorderStyle := bsNone;
  PrintLeft := 0;
  PrintTop := 0;
  PrintHeight := 1000;
  PrintWidth := 1000;
  FLineColor := clBlue;
  FSelectMode := smManual;
end;

{ **** Print the graph in all its glory **** }
procedure TLineGraph.PrintGraph;
var
  StartX, EndX, StepX,
  HPixelsPerValue, VPixelsPerValue : Single;

begin
  Inherited PrintGraph;                                    { Print everything but the data lines  }
  GetCalibration(VPixelsPerValue, HPixelsPerValue, StartX, EndX, StepX);
  PrintDataLines(VPixelsPerValue, HPixelsPerValue);                        { Print the data lines }
end;

{ **** Stop the graph from redrawing **** }
procedure TLineGraph.DisableGraph;
begin
  RePaintGraph := False;
end;

{ **** Re enable the graph for redrawing **** }
procedure TLineGraph.EnableGraph;
begin
  RePaintGraph := True;
  RePaint;
end;

procedure TLineGraph.ZoomIn;
begin
  Inherited ZoomIn;
end;

procedure TLineGraph.ZoomOut;
begin
  Inherited ZoomOut;
end;

procedure TLineGraph.SelectPixels(SelRect : TRect);
begin
  Inherited SelectPixels( SelRect );
end;

procedure TLineGraph.SelectValues(XMin, XMax, YMin, YMax : Double);
begin
  Inherited SelectValues(XMin, XMax, YMin, YMax);
end;

procedure TLineGraph.ClearSelection;
begin
  Inherited ClearSelection;
end;

{ ************************************************** }
{ ************* Miscalaneous Functions ************* }
{ ************************************************** }
function Max(Value1, Value2 : Extended) : Extended;
begin
  If Value1 > Value2 then
    Result := Value1
  else
    Result := Value2;
end;

function Min(Value1, Value2 : Extended) : Extended;
begin
  If Value1 < Value2 then
    Result := Value1
  else
    Result := Value2;
end;

function Exp10(Value : Double; Multiplier : Integer) : Double;
var
  Loop : Integer;

begin
  Result := Value;
  If Multiplier > 0 Then
    for Loop := 1 To Multiplier Do
      Result := Result * 10
  else
    if Multiplier < 0 Then
      for Loop := -1 DownTo Multiplier Do
        Result := Result / 10;
end;

{ ************************************************** }
{ ************* Registration procedure ************* }
{ ************************************************** }

procedure Register;
begin
  RegisterComponents('Extras', [TLineGraph]);  { Display the TLineGraph component on the tool bar }
end;

end.
